unit mainUnit;

{* Delphi Streaming Radio Library
 * Version v1.0.0
 * Copyright 2004-2007, Steve Blinch
 * http://code.blitzaffe.com
 * ============================================================================
 *
 * DESCRIPTION
 *
 * Provides a pure-Delphi implementation of a streaming radio client (to receive
 * MP3 audio from a streaming radio server) and server (to stream MP3 audio to
 * a streaming radio client such as WinAmp).  Full metadata support is included.
 *
 * The server has been most well-tested with IceCast but should work with
 * ShoutCast too.  The client has been tested with WinAmp and should work with
 * most other media players.
 *
 * Includes an example proxy application which takes a live audio stream from a
 * streaming server and relays it to one or more media players.
 *
 * Unfortunately this code is very poorly documented as it was part of a
 * personal project and never originally ntended for release, but hopefully the
 * example application is documentation enough in itself.
 *
 *
 * LICENSE
 *
 * This code is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *	
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *}

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, threadedICYClient, threadedICYServer, ComCtrls, rawbufferUnit,
  BusinessSkinForm, bsSkinCtrls, MMObj, MMBmpLst, MMBmpBtn, ExtCtrls,
  MMPanel, MMFill;

type
  TformStreaming = class(TForm)
    lblStatus: TLabel;
    bsBusinessSkinForm1: TbsBusinessSkinForm;
    MMPanelFill4: TMMPanelFill;
    lvClients: TListView;
    MMPanelFill5: TMMPanelFill;
    Label6: TLabel;
    MMPanelFill1: TMMPanelFill;
    memLog: TMemo;
    MMPanelFill2: TMMPanelFill;
    Label3: TLabel;
    MMPanelFill3: TMMPanelFill;
    Label1: TLabel;
    Label2: TLabel;
    edURL: TEdit;
    Label5: TLabel;
    edPort: TEdit;
    btnStart: TMMBitmapButton;
    bsSkinStatusBar1: TbsSkinStatusBar;
    procedure FormCreate(Sender: TObject);
    procedure btnStartClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    Started: Boolean;

    ICYClient: TThreadedICYClient;
    ICYServer: TThreadedICYServer;

    procedure SetCaption(Caption: String);
    procedure Log(Msg: String; Level: Integer);
    procedure ProcessICYBuffer(var Buffer: TRawBuffer);
    procedure FetchICYBuffer(var Buffer: TRawBuffer);
  public
    procedure WMUser(var Msg: TMessage); message WM_USER;
  end;

var
  formStreaming: TformStreaming;

implementation

{$R *.dfm}

uses threadlogUnit,wmCommUnit, form_mainform;

procedure TformStreaming.WMUser(var Msg: TMessage);
var
  ThreadMsg: PThreadMsg;
  Metadata: String;
  i: Integer;
  Item: TListItem;
  ICYServerThread: TICYServerThread;
  Buffer: TRawBuffer;

begin
  case Msg.WParam of

{ ***************************** Logging Messages ***************************** }


      // Called when a thread needs to make a note in the log; this is always accompanied
      // by a PThreadMsg containing the log information
      MSG_THREADLOG:
        begin
          ThreadMsg:=PThreadMsg(Msg.LParam);
          Log(ThreadMsg^.Buffer.AsString,ThreadMsg^.LogLevel);
          ThreadMsg^.Buffer.Free;
          Dispose(ThreadMsg);
        end;


{ **************************** ICY Client Messages *************************** }


      // Called when an ICY Client Connection is terminating
      MSG_ICYCLIENT_FREE:
        begin
          if (Started) then btnStart.OnClick(self);
          Log('Disconnected from streaming server',LL_NORMAL);
          SetCaption('Source stream disconnected');
        end;

      // Called when an ICY Client Connection is connected to a streaming server
      MSG_ICYCLIENT_CONNECTED:
        begin
          Log('Connected to streaming server',LL_NORMAL);
          SetCaption('Connected to stream');
        end;

      // Called when an ICY Client Connection has successfully negotiated an MPEG audio stream
      MSG_ICYCLIENT_STREAMING:
        begin
          Buffer:=wmReceiveBuffer(Msg);
          Buffer.Fetch(ICYServer.StreamInfo,SizeOf(TStreamInfo));
          Buffer.Free;
          
          Log('Receiving stream from streaming server',LL_NORMAL);
          SetCaption('Receiving stream: '+ICYServer.StreamInfo.Name);
        end;

      // Called when an ICY Client Connection has received a buffer full of MPEG audio data;
      // this is always accompanied by a TRawBuffer containing the MPEG audio data
      MSG_ICYCLIENT_BUFFER:
        begin
          Buffer:=wmReceiveBuffer(Msg);

          if not Buffer.Allocated then begin messagedlg('buffer not allocated',mterror,[mbok],0); exit; end;
          if Buffer.Size=0 then  begin messagedlg('buffer size is zero',mterror,[mbok],0); exit; end;
          
          // note that ProcessICYBuffer migrates the data out of Buffer, so Buffer.Buffer is
          // no longer allocated after this call
          ProcessICYBuffer(Buffer);

          // pass this to the server threads
          Buffer.Free;
        end;

      // Called when an ICY Client Connection receives stream metadata; this is always
      // accompanied by a TRawBuffer containing the metadata string
      MSG_ICYCLIENT_METADATA:
        begin
          MetaData:=wmReceiveString(Msg);
          ICYServer.SetMetaData(MetaData);
          Log('ICY-Meta received: ['+MetaData+']',LL_NORMAL);
        end;


{ **************************** ICY Server Messages *************************** }


      // Called when a client has connected to our ICY Server
      MSG_ICYSERVER_CONNECTED:
        begin
          ICYServerThread:=TICYServerThread(Msg.LParam);
          Item:=lvClients.Items.Add;
          Item.SubItems.Add('');
          Item.Caption:=ICYServerThread.Connection.Socket.Binding.PeerIP;
          Item.Data:=ICYServerThread;
        end;

      // Called when an ICY Server thread has detected that it has outbound data
      // pending; expects that the TRawBuffer in Msg.LParam will be filled with
      // the next block of data to be sent
      MSG_ICYSERVER_FETCH_BUFFER:
        begin
          Buffer:=TRawBuffer(Msg.LParam);
          FetchICYBuffer(Buffer);
        end;

      // Called when an ICY Server thread is disconnected and freed
      MSG_ICYSERVER_DISCONNECTED:
        begin
          ICYServerThread:=TICYServerThread(Msg.LParam);
          for i:=0 to lvClients.Items.Count-1 do
            begin
              Item:=lvClients.Items[i];
              if Item.Data=ICYServerThread then
                begin
                  lvClients.Items.Delete(i);
                  break;
                end;
            end;

        end;
  end;
end;

// processes a buffer received from an ICY client connection
procedure TformStreaming.ProcessICYBuffer(var Buffer: TRawBuffer);
begin
  if not Buffer.Allocated then exit;

  // stream data to our own local ICY server
  ICYServer.StreamData(Buffer);
end;

// ICY server threads call this method (via window message) to poll for outgoing
// data to be sent.
//
// We store a TList in the thread class itself containing the outbound blocks;
// to remain threadsafe, we only access that tlist from the main thread.
procedure TformStreaming.FetchICYBuffer(var Buffer: TRawBuffer);
var
  ICYThread: TICYServerThread;
  poppedBuffer: TRawBuffer;
begin
//  Log('Main::FetchICYBuffer() Checking outbound data buffer',LL_DEBUG);

  ICYThread:=Buffer.Peer as TICYServerThread;
  if ICYThread.OutboundBlocks.Count=0 then
    begin
      Log('Main::getHTTPOutboundData() No data!',LL_DEBUG);
      exit;
    end;

  poppedBuffer:=ICYThread.OutboundBlocks[0];

//  Log('Main::FetchICYBuffer() Popped buffer is '+IntToStr(poppedBuffer.Size)+' bytes',LL_DEBUG);

  ICYThread.OutboundBlocks.Delete(0);
  ICYThread.OutboundPending:=ICYThread.OutboundBlocks.Count>0;


  poppedBuffer.MigrateTo(Buffer);
  poppedBuffer.Free;
end;

procedure TformStreaming.FormCreate(Sender: TObject);
begin
  Started:=False;
  ICYServer:=TThreadedICYServer.Create(Self);
end;

procedure TformStreaming.btnStartClick(Sender: TObject);
begin
  Started:=not Started;
  ICYServer.DefaultPort:=StrToIntDef(edPort.Text,8000);
  ICYServer.Active:=Started;

  if (Started) then
    begin
      btnStart.Caption:='Stop';

      ICYClient:=TThreadedICYClient.Create(True,edURL.Text);
      ICYClient.FreeOnTerminate:=True;
      ICYClient.WindowHandle:=Handle;
      ICYClient.Resume;
    end
    else
    begin
      btnStart.Caption:='Start';

      ICYClient.Terminate;
    end;

    mainform.startupsetting.Lines.Strings[7]:=edurl.Text;
    mainform.startupsetting.Lines.Strings[8]:=edport.Text;
    mainform.startupsetting.Lines.SaveToFile(savedir+'setting\startup.dat');
end;

procedure TformStreaming.SetCaption(Caption: String);
begin
  lblStatus.Caption:=Caption;
end;

procedure TformStreaming.Log(Msg: String; Level: Integer);
begin
  memLog.Lines.Add(Msg);
end;

procedure TformStreaming.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  width:=0;
  height:=0;
  formStreaming.FormStyle:=fsnormal;
  formStreaming.Visible:=false;
  mainform.AutoArrange1.Click;
end;

end.
